<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Products extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'products';
    protected $primaryKey = 'product_id';

    protected $fillable = [
        'sku',
        'slug',
        'name',
        'type',
        'description',
        'description_short',
        'description_list',
        'price',
        'discount_price',
        'discount_type',
        'stock',
        'sold_out',
        'sold_count',
        'is_active',
        'is_featured',
        'is_new',
        'category_id',
        'brand_id',
        'weight',
        'length',
        'width',
        'height',
        'unit',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'discount_price' => 'decimal:2',
        'weight' => 'decimal:2',
        'length' => 'decimal:2',
        'width'  => 'decimal:2',
        'height' => 'decimal:2',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'is_new' => 'boolean',
        'sold_out' => 'boolean',
        'description_list' => 'array',
    ];

    /* ===============================
     *          RELATIONS
     * =============================== */

    // Category level 1
    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id', 'category_id');
    }

    // Brand
    public function brand()
    {
        return $this->belongsTo(Brand::class, 'brand_id', 'brand_id');
    }

    // Creator & Editor
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by', 'id');
    }
    public function editor()
    {
        return $this->belongsTo(User::class, 'updated_by', 'id');
    }

    // Polymorphic SEO
    public function seometa()
    {
        return $this->morphOne(SeoMeta::class, 'seoble');
    }

    // Polymorphic Media
    public function media()
    {
        return $this->morphMany(Media::class, 'mediable');
    }


    /* ===============================
     *          SCOPES
     * =============================== */

    // Hanya produk aktif
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
    // Produk featured
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }
    // Search by name or SKU
    public function scopeSearch($query, $term)
    {
        return $query->where(function ($q) use ($term) {
            $q->where('name', 'like', "%$term%")
                ->orWhere('sku', 'like', "%$term%");
        });
    }

    /* ===============================
     *      HELPER METHODS
     * =============================== */

    public function finalPrice()
    {
        if ($this->discount_price) {
            return $this->discount_price;
        }
        return $this->price;
    }
    public function hasDiscount()
    {
        return !empty($this->discount_price);
    }
}
