<?php

namespace App\Http\Livewire\Setting;

use App\Models\Menu;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;
use Livewire\Attributes\Title;

#[Layout('components.layouts.panel-admin')]
#[Title('Menu dashboard peronit')]
class SettingMenuAdmin extends Component
{

    use WithPagination;

    public $search, $pages = 15;

    public $sortField = 'menu_id';
    public $sortDirection = 'asc';
    public $selected = [];
    public $selectAll = false;
    public $def;

    // Data untuk create
    public $field = [
        'name' => '',
        'url' => '',
        'route' => '',
        'icon' => '',
        'parent_id' => null,
        'type' => 'item',
        'sort_order' => 0,
        'is_active' => true,
    ];

    protected $rules = [
        'field.type' => 'string|max:255',
        'field.name' => 'string|max:255',
        'field.url' => 'nullable|string|max:255',
        'field.route' => 'nullable|string|max:255',
        'field.icon' => 'nullable|string',
        'field.parent_id' => 'nullable|exists:menus,menu_id',
        'field.is_active' => 'boolean',
    ];

    protected $messages = [
        'field.name.string'   => 'Nama menu harus berupa teks.',
        'field.name.max'      => 'Nama menu maksimal 255 karakter.',
        'field.url.string'   => 'url harus berupa teks.',
        'field.url.max'      => 'url maksimal 255 karakter.',
        'field.route.string'   => 'route harus berupa teks.',
        'field.route.max'      => 'route maksimal 255 karakter.',
        'field.icon.string' => 'Deskripsi harus berupa teks.',
        'field.parent_id.exists'   => 'Menu induk yang dipilih tidak valid.',
        'field.is_active.boolean'  => 'Status aktif harus berupa true/false.',
    ];

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    /* ============ BULK SELECT ============ */
    public function updatingSelectAll($value)
    {
        if ($value) {
            $this->selected = Menu::pluck('menu_id')->toArray();
        } else {
            $this->selected = [];
        }
    }

    public function updatedSelected()
    {
        $this->selectAll = false;
    }

    public function deleteSelected()
    {
        Menu::whereIn('menu_id', $this->selected)->delete();
        Menu::whereIn('parent_id', $this->selected)->delete();
        $this->selected = [];
        $this->selectAll = false;
    }

    public function delete($id)
    {
        Menu::findOrFail($id)->delete();
    }

    public function submitedMenu()
    {
        $this->validate();

        Menu::updateOrCreate(
            ['menu_id' => $this->field['menu_id'] ?? null], // kondisi
            [
                'name' => $this->field['name'],
                'type' => $this->field['type'],
                'url' => $this->field['url'],
                'route' => $this->field['route'],
                'icon' => $this->field['icon'],
                'parent_id' => $this->field['parent_id'],
                'sort_order' => $this->field['sort_order'],
                'is_active' => $this->field['is_active'],
            ]
        );

        $this->resetForm();
        $this->dispatch('success', 'Category created successfully!');
        $this->dispatch('closeModal', id: 'MenuModal');
    }

    protected function resetForm()
    {
        $this->reset('field');
        $this->field = [
            'name' => '',
            'url' => '',
            'route' => '',
            'icon' => '',
            'parent_id' => null,
            'type' => 'item',
            'sort_order' => 0,
            'is_active' => true,
        ];
    }

    // Add new sub category
    public function addMenu()
    {
        $this->resetForm();
        $this->dispatch('openModal', id: 'MenuModal');
    }

    // Add new sub category
    public function addSubMenu($parentId)
    {
        $this->resetForm();
        $this->field['parent_id'] = $parentId;
        $this->dispatch('openModal', id: 'MenuModal');
    }

    // Add new Child Category
    public function addChildMenu($parentId)
    {
        $this->resetForm();
        $this->field['parent_id'] = $parentId;
        $this->dispatch('openModal', id: 'MenuModal');
    }

    public function edit($id)
    {
        $this->resetForm();
        $this->field = Menu::find($id)->toArray();
        $this->dispatch('openModal', id: 'MenuModal');
    }

    public function render()
    {
        $menu = Menu::with('parent', 'children')
            ->Search($this->search)
            ->orderBy('sort_order')
            ->paginate($this->pages);

        $allMenu = Menu::orderBy('name')->get();
        return view('setting.setting-menu-admin', [
            'menu' => $menu,
            'allMenu' => $allMenu,
        ]);
    }
}
