<?php

namespace App\Http\Livewire\Products\ProductServices;

use App\Traits\WithDescriptionList;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Media;
use App\Models\Products;
use App\Models\SeoMeta;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Title;

#[Layout('components.layouts.panel-admin')]
#[Title('Edit product')]
class ProductServicesEditAdmin extends Component
{
    use WithFileUploads, WithDescriptionList;

    public $id, $product_id;

    // Form fields
    public $product = [], $seometa = [];

    // Relations
    public $category, $brands;

    protected $rules =  [
        'product.sku' => 'required:max:255',
        'product.name' => 'required:max:255',
        'product.price' => 'required:number',
        'product.type' => 'required',
        'product.is_active' => 'required',
        'product.description_short' => 'required',
    ];

    protected $messages = [
        'product.name.required' => 'Nama product tidak boleh kosong!',
        'product.price.required' => 'Harga tidak boleh kosong!',
        'product.price.integer' => 'Harga bukan number!',
        'product.type.required' => 'Type tidak boleh kosong!',
        'product.status.required' => 'Status tidak boleh kosong!',
        'product.description_short.required' => 'Description tidak boleh kosong!'
    ];

    public function mount()
    {
        // Prepare data
        $data = Products::with('media')->findOrFail($this->id);
        $this->product = $data->toArray();
        $this->priceFormatted = number_format($data['price'], 0, ',', '.');
        $this->product['images'] = $data->media()->first() ?? null;
        $this->description_list = $this->product['description_list'];
        $this->seometa = $data->seometa()->first()->toArray();

        $this->category = Category::with('children.children')->whereNull('parent_id')->get();
        $this->brands     = Brand::all();
    }

    public function save()
    {
        $this->validate();

        $product = Products::findOrFail($this->id);

        // Update product fields
        $product->update([
            'name'               => $this->product['name'],
            'slug'               => $this->product['slug'] ?: Str::slug($this->product['name']),
            'price'              => $this->product['price'],
            'type'               => $this->product['type'],
            'description'        => $this->product['description'],
            'description_short'  => $this->product['description_short'],
            'description_list'   => $this->description_list,
            'is_active'          => $this->product['is_active'],
            'category_id'        => $this->product['category_id'],
            'brand_id'           => $this->product['brand_id'],
            'updated_by'         => null,
        ]);

        // Relation
        if (is_object($this->product['images']) && method_exists($this->product['images'], 'temporaryUrl')) {
            $this->validate([
                'product.images' => 'sometimes|file|max:12288|mimes:jpg,jpeg,png,sgv',
            ], [
                'product.images.image' => 'Oops, sepertinya yang diupload bukan gambar!',
                'product.images.mimes' => 'Format gambar harus jpeg, png, jpg, gif, atau svg!',
                'product.images.max' => 'Ukuran gambar maksimal 4MB!',
            ]);
            $product->media()->delete();
            Media::saveMedia($product, $this->product['images'], 'PRDSV', true);
        }
        SeoMeta::saveSeoMeta($product, $this->seometa['meta_title'], $this->seometa['meta_description'], $this->seometa['meta_keywords'], $this->seometa['meta_image']);

        $this->dispatch('success', 'Produk berhasil diperbarui!');
        return redirect()->route('admin.product.services');
    }

    // FORMATED
    public $priceFormatted;
    public function formatPrice()
    {
        // Hapus semua karakter kecuali angka
        $numeric = preg_replace('/[^0-9]/', '', $this->priceFormatted);
        // Simpan sebagai integer asli
        $this->product['price'] = $numeric;
        $this->priceFormatted = number_format($numeric, 0, ',', '.');
    }

    public function render()
    {
        return view('products.productServices.product-services-edit-admin');
    }
}
