<?php

namespace App\Http\Livewire\Products\ProductServices;

use App\Helpers\HelperGenerateSku;
use App\Traits\WithDescriptionList;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Media;
use App\Models\Products;
use App\Models\SeoMeta;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Title;
use Livewire\WithFileUploads;

#[Layout('components.layouts.panel-admin')]
#[Title('Create new product')]
class ProductServicesCreateAdmin extends Component
{

    use WithFileUploads, WithDescriptionList;

    // Form fields
    public $product = [
        'sku'                => null,
        'name'               => null,
        'price'              => 0,
        'type'               => null,
        'description'        => null,
        'description_short'  => null,
        'is_active'          => true,
        'created_by'         => null,
        'category_id'        => null,
        'brand_id'           => 0,
        'type'               => 'services',
        'images'              => null
    ], $seometa = [
        'meta_title' => '',
        'meta_keywords' => '',
        'meta_description' => '',
    ];


    // Relations
    public $category = [];
    public $brands = [];

    public function save()
    {
        $this->validate([
            'product.sku' => 'required:max:255',
            'product.name' => 'required:max:255',
            'product.price' => 'required:number',
            'product.type' => 'required',
            'product.is_active' => 'required',
            'product.description_short' => 'required',
            'product.images' => 'required|file|max:12288|mimes:jpg,jpeg,png,sgv',
            'product.category_id'       => 'required|exists:categories,category_id',
            'product.brand_id'          => 'nullable|exists:brands,brand_id',
        ], [
            'product.sku.required' => 'SKU product tidak boleh kosong!',
            'product.name.required' => 'Nama product tidak boleh kosong!',
            'product.price.required' => 'Harga tidak boleh kosong!',
            'product.price.integer' => 'Harga bukan number!',
            'product.type.required' => 'Type tidak boleh kosong!',
            'product.status.required' => 'Status tidak boleh kosong!',
            'product.description_short.required' => 'Description tidak boleh kosong!',
            'product.images.required' => 'Image tidak boleh kosong!',
            'product.images.file' => 'File upload harus berupa file image',
            'product.images.mimes' => 'File upload format tidak sah!',
            'product.images.max' => 'File upload melebihi kapasitas!',
        ]);

        $product = Products::create([
            'sku'                => $this->product['sku'],
            'name'               => $this->product['name'],
            'slug'               => Str::slug($this->product['name']),
            'price'              => $this->product['price'],
            'type'               => $this->product['type'],
            'description'        => $this->product['description'],
            'description_short'  => $this->product['description_short'],
            'description_list'   => $this->description_list,
            'is_active'          => $this->product['is_active'],
            'created_by'         => null,
            'category_id'        => $this->product['category_id'],
            'brand_id'           => $this->product['brand_id'],
            'type'               => 'services',
            // 'created_by'         => Auth::id(),
        ]);

        // Relation
        Media::saveMedia($product, $this->product['images'], 'PRDSV', true);
        SeoMeta::saveSeoMeta($product, $this->seometa['meta_title'], $this->seometa['meta_description'], $this->seometa['meta_keywords'], null);

        $this->dispatch('success', 'Product berhasil ditambahkan!');
        return redirect()->route('admin.product.services');
    }

    public function mount()
    {
        $this->category = Category::with('children.children')->whereNull('parent_id')->get();
        $this->brands     = Brand::all();
        $this->product['sku'] = HelperGenerateSku::generate('products', 'PRDSV');
    }


    // FORMATED
    public $priceFormatted;
    public function formatPrice()
    {
        // Hapus semua karakter kecuali angka
        $numeric = preg_replace('/[^0-9]/', '', $this->priceFormatted);
        // Simpan sebagai integer asli
        $this->product['price'] = $numeric;
        $this->priceFormatted = number_format($numeric, 0, ',', '.');
    }

    public function render()
    {
        return view('products.productServices.product-services-create-admin');
    }
}
