<?php

namespace App\Http\Livewire\Products\ProductCategory;

use App\Traits\WithBulkDeleteActions;
use App\Models\Category;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;
use Livewire\Attributes\Title;

#[Layout('components.layouts.panel-admin')]
#[Title('Category product')]
class ProductCategoryAdmin extends Component
{
    use WithPagination, WithBulkDeleteActions;

    protected string $modelClass = Category::class;

    public $sortField = 'category_id';
    public $sortDirection = 'asc';

    // Data untuk create
    public $categoryField = [
        'name' => '',
        'slug' => '',
        'description' => '',
        'parent_id' => null,
        'type' => null,
        'is_active' => true,
    ];

    protected $rules = [
        'categoryField.type' => 'required|string|max:255',
        'categoryField.name' => 'required|string|max:255',
        'categoryField.slug' => 'nullable|string|max:255',
        'categoryField.description' => 'nullable|string',
        'categoryField.parent_id' => 'nullable|exists:categories,category_id',
        'categoryField.is_active' => 'boolean',
    ];

    protected $messages = [
        'categoryField.type.required' => 'Type kategori wajib diisi.',
        'categoryField.name.required' => 'Nama kategori wajib diisi.',
        'categoryField.name.string'   => 'Nama kategori harus berupa teks.',
        'categoryField.name.max'      => 'Nama kategori maksimal 255 karakter.',
        'categoryField.slug.string'   => 'Slug harus berupa teks.',
        'categoryField.slug.max'      => 'Slug maksimal 255 karakter.',
        'categoryField.slug.unique'   => 'Slug ini sudah dipakai, gunakan slug lain.',
        'categoryField.description.string' => 'Deskripsi harus berupa teks.',
        'categoryField.parent_id.exists'   => 'Kategori induk yang dipilih tidak valid.',
        'categoryField.is_active.boolean'  => 'Status aktif harus berupa true/false.',
    ];

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function delete($id)
    {
        Category::findOrFail($id)->delete();
    }

    public function submitedCategory()
    {
        $this->validate();

        Category::updateOrCreate(
            ['category_id' => $this->categoryField['category_id'] ?? null], // kondisi
            [
                'name' => $this->categoryField['name'],
                'type' => $this->categoryField['type'],
                'slug' => Str::slug($this->categoryField['name']),
                'description' => $this->categoryField['description'],
                'parent_id' => $this->categoryField['parent_id'],
                'is_active' => $this->categoryField['is_active'],
            ]
        );

        $this->resetForm();
        $this->dispatch('success', 'Category created successfully!');
        $this->dispatch('closeModal', id: 'CategoryModal');
    }

    protected function resetForm()
    {
        $this->reset('categoryField');
        $this->categoryField = [
            'name' => '',
            'slug' => '',
            'description' => '',
            'parent_id' => null,
            'is_active' => true,
        ];
    }

    // Add new sub category
    public function addCategory()
    {
        $this->resetForm();
        $this->dispatch('openModal', id: 'CategoryModal');
    }

    // Add new sub category
    public function addSubCategory($parentId)
    {
        $this->resetForm();
        $this->categoryField['parent_id'] = $parentId;
        $this->dispatch('openModal', id: 'CategoryModal');
    }

    // Add new Child Category
    public function addChildCategory($parentId)
    {
        $this->resetForm();
        $this->categoryField['parent_id'] = $parentId;
        $this->dispatch('openModal', id: 'CategoryModal');
    }

    public function edit($id)
    {
        $this->resetForm();
        $this->categoryField = Category::find($id)->toArray();
        $this->dispatch('openModal', id: 'CategoryModal');
    }

    public function render()
    {
        $categories = Category::with('parent', 'children')
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate(10);

        $allCategories = Category::orderBy('name')->get();

        return view('products.productCategory.product-category-admin', [
            'categories' => $categories,
            'allCategories' => $allCategories,
        ]);
    }
}
