<?php

namespace App\Http\Livewire\Products\ProductBrand;

use App\Models\Brand;
use App\Models\Media;
use App\Traits\WithBulkDeleteActions;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\WithFileUploads;

#[Layout('components.layouts.panel-admin')]
#[Title('Brand product')]
class ProductBrandAdmin extends Component
{
    use WithPagination, WithFileUploads, WithBulkDeleteActions;

    protected string $modelClass = Brand::class;


    public $sortField = 'brand_id';
    public $sortDirection = 'asc';
    public $selected = [];
    public $selectAll = false;
    public $pages = 15;

    // Data untuk create
    public $field = [
        'name' => '',
        'slug' => '',
        'logo' => null,
        'description' => '',
        'type' => '',
        'is_active' => true,
    ];

    protected $rules = [
        'field.type' => 'required|string|max:255',
        'field.name' => 'required|string|max:255',
        'field.slug' => 'nullable|string|max:255',
        'field.description' => 'nullable|string',
        'field.is_active' => 'boolean',
        'field.logo' => 'required|file|max:12288|mimes:jpg,jpeg,png,sgv',

    ];

    protected $messages = [
        'field.name.type'     => 'Type kategori wajib diisi.',
        'field.name.required' => 'Nama kategori wajib diisi.',
        'field.name.string'   => 'Nama kategori harus berupa teks.',
        'field.name.max'      => 'Nama kategori maksimal 255 karakter.',
        'field.slug.string'   => 'Slug harus berupa teks.',
        'field.slug.max'      => 'Slug maksimal 255 karakter.',
        'field.slug.unique'   => 'Slug ini sudah dipakai, gunakan slug lain.',
        'field.description.string' => 'Deskripsi harus berupa teks.',
        'field.is_active.boolean'  => 'Status aktif harus berupa true/false.',
        'field.logo.required' => 'Image tidak boleh kosong!',
        'field.logo.file' => 'File upload harus berupa file image',
        'field.logo.mimes' => 'File upload format tidak sah!',
        'field.logo.max' => 'File upload melebihi kapasitas!',
    ];

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function delete($id)
    {
        Brand::findOrFail($id)->delete();
    }

    public function submitedBrand()
    {
        $this->validate();

        $brand = Brand::updateOrCreate(
            ['brand_id' => $this->field['brand_id'] ?? null], // kondisi
            [
                'name' => $this->field['name'],
                'type' => $this->field['type'],
                'slug' => Str::slug($this->field['name']),
                'description' => $this->field['description'],
                'is_active' => $this->field['is_active'],
            ]
        );

        Media::saveMedia($brand, $this->field['logo'], 'BRN', true);

        $this->resetForm();
        $this->dispatch('success', 'Category created successfully!');
        $this->dispatch('closeModal', id: 'CategoryModal');
    }

    protected function resetForm()
    {
        $this->reset('field');
        $this->field = [
            'name' => '',
            'slug' => '',
            'logo' => null,
            'description' => '',
            'type' => '',
            'is_active' => true,
        ];
    }

    // Add new sub category
    public function addBrand()
    {
        $this->resetForm();
        $this->dispatch('openModal', id: 'CategoryModal');
    }

    public function edit($id)
    {
        $this->resetForm();
        $this->field = Brand::find($id)->toArray();
        $this->dispatch('openModal', id: 'CategoryModal');
    }
    public function render()
    {
        $data = Brand::paginate($this->pages);
        return view('products.productBrand.product-brand-admin', [
            'brand' => $data
        ]);
    }
}
