<?php

namespace App\Http\Livewire\Content;

use App\Traits\WithBulkDeleteActions;
use App\Models\ContentBanner;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Livewire\WithFileUploads;
use Livewire\WithPagination;

#[Layout('components.layouts.panel-admin')]
class ContentBannersAdmin extends Component
{

    use WithFileUploads, WithPagination, WithBulkDeleteActions;

    protected string $modelClass = ContentBanner::class;

    public $sortField = 'banner_id';
    public $sortDirection = 'asc';

    public $pages = 15, $search;

    public $form = [
        'banner_id' => null,
        'title' => '',
        'description' => '',
        'link' => '',
        'link_text' => '',
        'image_path' => '',
        'image_desktop' => null,
        'image_mobile' => null,
        'text_position' => '',
        'text_color' => '',
        'is_active' => 1,
        'order' => 0,
        'page' => '',
        'start_date' => '',
        'end_date' => ''
    ];

    protected $rules = [
        'form.image_desktop' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:4096',
        'form.image_mobile' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:4096',
    ];

    protected $messages = [
        'form.image_desktop.required' => 'Oops, please input banner desktop!',
        'form.image_desktop.image' => 'Oops, sepertinya yang diupload bukan gambar!',
        'form.image_desktop.mimes' => 'Format gambar harus jpeg, png, jpg, gif, atau svg!',
        'form.image_desktop.max' => 'Ukuran gambar maksimal 4MB!',

        'form.image_mobile.required' => 'Oops, please input banner mobile!',
        'form.image_mobile.image' => 'Oops, sepertinya yang diupload bukan gambar!',
        'form.image_mobile.mimes' => 'Format gambar harus jpeg, png, jpg, gif, atau svg!',
        'form.image_mobile.max' => 'Ukuran gambar maksimal 4MB!',
    ];

    public function show($id)
    {
        $this->form = ContentBanner::find($id)->toArray();
        $this->dispatch('openModal', id: 'modalPreview');
    }

    public function new()
    {
        $this->reset('form');
        $this->dispatch('openModal', id: 'modalBanner');
    }

    public function edit($id)
    {
        $this->form = ContentBanner::find($id)->toArray();
        $this->dispatch('openModal', id: 'modalBanner');
    }

    public function delete($id) 
    {
        $this->form = ContentBanner::find($id)->toArray();
        $this->dispatch('openModal', id: 'modalDelete');
    }

    public function deleteAction()
    {
        ContentBanner::findOrFail($this->form['banner_id'])->delete();
        $this->dispatch('hiddenModal', 'modalDelete');
        $this->dispatch('succes', 'Data berhasil di hapus!');
    }

    public function store()
    {
        if ($this->form['image_desktop'] and $this->form['image_mobile'] instanceof TemporaryUploadedFile) {
            $this->validate();
        }

        $imagePath = 'images/pages/banner/';

        // DESKTOP
        if ($this->form['image_desktop'] instanceof TemporaryUploadedFile) {
            $desktopName = 'BNR-D-' . md5(now()) . '.' . $this->form['image_desktop']->getClientOriginalExtension();
            $this->form['image_desktop']->storeAs($imagePath, $desktopName, 's4');
        } else {
            $desktopName = $this->form['image_desktop']; // lama
        }

        // MOBILE
        if ($this->form['image_mobile'] instanceof TemporaryUploadedFile) {
            $mobileName = 'BNR-M-' . md5(now()) . '.' . $this->form['image_mobile']->getClientOriginalExtension();
            $this->form['image_mobile']->storeAs($imagePath, $mobileName, 's4');
        } else {
            $mobileName = $this->form['image_mobile']; // lama
        }

        // SIMPAN
        $data = ContentBanner::updateOrCreate(
            ['banner_id' => $this->form['banner_id']],
            [
                'title'         => $this->form['title'],
                'description'   => $this->form['description'],
                'link'          => $this->form['link'],
                'link_text'     => $this->form['link_text'],
                'image_path'    => $imagePath,
                'image_desktop' => $desktopName,
                'image_mobile'  => $mobileName,
                'text_position' => $this->form['text_position'],
                'text_color'    => $this->form['text_color'],
                'is_active'     => $this->form['is_active'],
                'order'         => $this->form['order'],
                'page'          => $this->form['page'],
                'start_date'    => $this->form['start_date'],
                'end_date'      => $this->form['end_date'],
            ]
        );

        if ($data) {
            if ($this->form['image_desktop'] instanceof TemporaryUploadedFile) {
                $this->form['image_desktop']->storeAs($imagePath, $desktopName, 's4');
            }
            if ($this->form['image_mobile'] instanceof TemporaryUploadedFile) {
                $this->form['image_mobile']->storeAs($imagePath, $mobileName, 's4');
            }
            $this->dispatch('success', 'Data banner telah ditambahkan');
            $this->dispatch('hiddenModal', 'modalBanner');
        } else {
            $this->dispatch('error', 'Oops, terjadi kesalahan pada database. Coba lagi nanti!');
        }
    }

    public function render()
    {
        $data = ContentBanner::orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->pages);
        return view('content.content-banners-admin', [
            'data' => $data
        ]);
    }
}
