<?php

namespace App\Http\Livewire\Account\Profile;

use App\Models\Admin;
use App\Models\Media;
use Livewire\Component;
use Livewire\WithFileUploads;

class ProfileAdminForm extends Component
{
    use WithFileUploads;

    public $admin_id, $admin = [];
    public $edit = false;

    protected $rules = [
        'admin.username' => 'required|min:3|max:50',

        'admin.phone' => [
            'required',
            'numeric',
            'digits_between:9,12',    // nomor Indonesia biasanya 9–12 digit
            'regex:/^(08)/',          // harus mulai dari 08 (standar nomor Indo)
        ],

        'admin.born'   => 'required|date',
        'admin.gender' => 'required|in:M,F,O',
    ];

    protected $messages = [
        // USERNAME
        'admin.username.required' => 'Username wajib diisi.',
        'admin.username.min'      => 'Username minimal harus 3 karakter.',
        'admin.username.max'      => 'Username maksimal 50 karakter.',

        // PHONE
        'admin.phone.required'       => 'Nomor telepon wajib diisi.',
        'admin.phone.numeric'        => 'Nomor telepon hanya boleh berisi angka.',
        'admin.phone.digits_between' => 'Nomor telepon harus antara 9 sampai 12 digit.',
        'admin.phone.regex'          => 'Nomor telepon harus dimulai dengan angka 08.',

        // BORN
        'admin.born.required' => 'Tanggal lahir wajib diisi.',
        'admin.born.date'     => 'Format tanggal lahir tidak valid.',

        // GENDER
        'admin.gender.required' => 'Jenis kelamin wajib dipilih.',
        'admin.gender.in'       => 'Jenis kelamin tidak valid.',
    ];


    public function mount()
    {
        $data = Admin::find($this->admin_id);
        $this->admin = $data->toArray();
        $this->admin['images'] = $data->media()->first() ?? null;
    }


    public function edited()
    {
        $this->edit = true;
    }

    public function cancle()
    {
        $this->edit = false;
    }

    public function save()
    {
        try {
            //code...
            $this->validate();
            $data = Admin::find($this->admin_id);
            $data->username = $this->admin['username'];
            $data->phone = $this->admin['phone'];
            $data->gender = $this->admin['gender'];
            $data->born = $this->admin['born'];
            $data->save();
            $this->dispatch('profile-refresh');
            $this->dispatch('success', 'Data berhasil diperbaharui!');
        } catch (\Throwable $th) {
            //throw $th;
            $this->dispatch('error', 'Database Error, data Gagal terupdate!!!');
        }
    }

    public function uploadImage()
    {
         try {
            //code...
            $this->validate(
                ['admin.images' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:4096'],
                [
                    'admin.images.required' => 'Foto wajib diunggah.',
                    'admin.images.image'    => 'File harus berupa gambar.',
                    'admin.images.mimes'    => 'Format gambar harus jpeg, png, jpg, gif, atau svg.',
                    'admin.images.max'      => 'Ukuran gambar maksimal 4MB.',
                ]);

            $admin = Admin::find($this->admin_id);
            Media::saveMedia($admin, $this->admin['images'], 'ADM', true);
            
            $this->dispatch('profile-refresh');
            $this->dispatch('success', 'Data berhasil diperbaharui!');
         } catch (\Throwable $th) {
            //throw $th;
            $this->dispatch('error', 'Database Error, data Gagal terupdate!!!');
         }
    }


    public function render()
    {
        return view('account.profile.profile-admin-form');
    }
}
